/*
  XIAO ESP32-C6 (BOTTOM) — Zigbee Coordinator Honeypot
  Logs devices attempting to join the network
  Author: Erik Wright
*/

#include <Arduino.h>
#include <ArduinoJson.h>
#include "esp_zigbee_core.h"
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"

// UART pins to communicate with TOP board
#define ZB_UART_RX   D7
#define ZB_UART_TX   D6
#define ZB_UART_BAUD 115200

// Zigbee endpoint configuration
#define HA_ESP_LIGHT_ENDPOINT 10
#define ESP_ZB_PRIMARY_CHANNEL_MASK (1l << 11)  // Use Zigbee channel 11

// Send a JSON document to the TOP board via Serial1
static void sendJson(const JsonDocument &doc){
  serializeJson(doc, Serial1);
  Serial1.print('\n');  // End with newline (one message per line)
  Serial1.flush();
}

// Send an event message to TOP board with optional extra data
static void sendEvt(const char* evt, std::function<void(JsonDocument&)> add = nullptr){
  StaticJsonDocument<512> d;
  d["evt"] = evt;  // Event type
  if (add) add(d); // Add extra fields if provided
  sendJson(d);
}

// Buffer for incoming commands from TOP board
static String inbuf;

// Process a command received from the TOP board
static void handleCommandLine(const char* line){
  StaticJsonDocument<512> d;
  DeserializationError err = deserializeJson(d, line);
  
  // If JSON parsing failed, send error
  if (err){
    sendEvt("error", [&](JsonDocument& j){ 
      j["msg"]="bad json"; 
      j["detail"]=err.c_str(); 
    });
    return;
  }
  
  // Get the command field
  const char* cmd = d["cmd"] | "";
  if (!*cmd){ 
    sendEvt("error", [&](JsonDocument& j){ j["msg"]="missing cmd"; }); 
    return; 
  }

  // Handle "permit join" command - opens network for devices to join
  if (strcmp(cmd, "zb_permit_join")==0){
    unsigned s = d["seconds"] | 60;  // Default 60 seconds
    esp_zb_bdb_open_network(s);      // Open Zigbee network
    sendEvt("zb_permit_join", [&](JsonDocument& j){ 
      j["seconds"]=s; 
      j["started"]=true; 
    });
  }
  // TODO: Add handlers for other commands (identify, on/off, etc.)
}

// Check Serial1 for incoming commands from TOP board
static void pumpSerial1(){
  while (Serial1.available()){
    char c = (char)Serial1.read();
    
    if (c == '\n'){  // End of line = complete command
      if (inbuf.length() > 0) {
        handleCommandLine(inbuf.c_str());
      }
      inbuf.clear();
    } else if (c != '\r'){  // Ignore carriage returns
      if (inbuf.length() < 700) inbuf += c;
    }
  }
}

// Main Zigbee task - runs in separate FreeRTOS task
static void esp_zb_task(void *pvParameters) {
  // Configure as Zigbee Coordinator (network hub)
  esp_zb_cfg_t zb_nwk_cfg = {
    .esp_zb_role = ESP_ZB_DEVICE_TYPE_COORDINATOR,  // Act as coordinator
    .install_code_policy = false,  // No security - allow any device (honeypot!)
    .nwk_cfg = {
      .zczr_cfg = {
        .max_children = 10,  // Allow up to 10 devices to join
      },
    },
  };
  esp_zb_init(&zb_nwk_cfg);
  
  // Create endpoint list (devices/services this coordinator offers)
  esp_zb_ep_list_t *esp_zb_ep_list = esp_zb_ep_list_create();
  esp_zb_cluster_list_t *esp_zb_cluster_list = esp_zb_zcl_cluster_list_create();
  
  // Add Basic cluster (device info)
  esp_zb_attribute_list_t *esp_zb_basic_cluster = esp_zb_basic_cluster_create(NULL);
  esp_zb_cluster_list_add_basic_cluster(esp_zb_cluster_list, esp_zb_basic_cluster, ESP_ZB_ZCL_CLUSTER_SERVER_ROLE);
  
  // Add Identify cluster (allows devices to "blink" for identification)
  esp_zb_attribute_list_t *esp_zb_identify_cluster = esp_zb_identify_cluster_create(NULL);
  esp_zb_cluster_list_add_identify_cluster(esp_zb_cluster_list, esp_zb_identify_cluster, ESP_ZB_ZCL_CLUSTER_SERVER_ROLE);
  
  // Add On/Off cluster (control light on/off)
  esp_zb_attribute_list_t *esp_zb_on_off_cluster = esp_zb_on_off_cluster_create(NULL);
  esp_zb_cluster_list_add_on_off_cluster(esp_zb_cluster_list, esp_zb_on_off_cluster, ESP_ZB_ZCL_CLUSTER_SERVER_ROLE);
  
  // Configure endpoint as a "Light Bulb" device
  esp_zb_endpoint_config_t endpoint_config = {
    .endpoint = HA_ESP_LIGHT_ENDPOINT,
    .app_profile_id = ESP_ZB_AF_HA_PROFILE_ID,          // Home Automation profile
    .app_device_id = ESP_ZB_HA_ON_OFF_LIGHT_DEVICE_ID,  // Pretend to be a light bulb
    .app_device_version = 0
  };
  
  // Register the endpoint with Zigbee stack
  esp_zb_ep_list_add_ep(esp_zb_ep_list, esp_zb_cluster_list, endpoint_config);
  esp_zb_device_register(esp_zb_ep_list);
  
  // Set which Zigbee channel to use (channel 11)
  esp_zb_set_primary_network_channel_set(ESP_ZB_PRIMARY_CHANNEL_MASK);
  
  // Start Zigbee stack and enter main loop
  ESP_ERROR_CHECK(esp_zb_start(false));
  esp_zb_stack_main_loop();  // This runs forever
}

// Handle Zigbee events (device joins, network status changes, etc.)
void esp_zb_app_signal_handler(esp_zb_app_signal_t *signal_struct) {
  uint32_t *p_sg_p = signal_struct->p_app_signal;
  esp_err_t err_status = signal_struct->esp_err_status;
  esp_zb_app_signal_type_t sig_type = (esp_zb_app_signal_type_t)*p_sg_p;
  
  switch (sig_type) {
    // Network started for first time or after reboot
    case ESP_ZB_BDB_SIGNAL_DEVICE_FIRST_START:
    case ESP_ZB_BDB_SIGNAL_DEVICE_REBOOT:
      if (err_status == ESP_OK) {
        esp_zb_bdb_open_network(255);  // Open network for 255 seconds
        sendEvt("zb_started", [](JsonDocument& j){
          j["msg"] = "Coordinator started - network open";
          j["channel"] = 11;
        });
      }
      break;
      
    // A device announced itself (joined the network) - THE HONEYPOT CATCHES THIS!
    case ESP_ZB_ZDO_SIGNAL_DEVICE_ANNCE: {
      esp_zb_zdo_signal_device_annce_params_t *dev_annce_params = 
        (esp_zb_zdo_signal_device_annce_params_t *)esp_zb_app_signal_get_params(p_sg_p);
      
      // Send device info to TOP board for logging
      sendEvt("zb_device_join", [dev_annce_params](JsonDocument& j){
        char nwk_str[8];
        snprintf(nwk_str, sizeof(nwk_str), "0x%04X", dev_annce_params->device_short_addr);
        j["nwk_addr"] = nwk_str;          // Device's network address
        j["capability"] = dev_annce_params->capability;  // Device capabilities
      });
      break;
    }
    
    // Network permit-join status changed
    case ESP_ZB_NWK_SIGNAL_PERMIT_JOIN_STATUS:
      if (err_status == ESP_OK) {
        sendEvt("zb_permit_status", [](JsonDocument& j){
          j["open"] = true;
        });
      }
      break;
      
    default:
      // Ignore other events
      break;
  }
}

void setup(){
  // Setup LED for blinking (shows board is alive)
  pinMode(LED_BUILTIN, OUTPUT);
  digitalWrite(LED_BUILTIN, LOW);
  
  // Setup debug serial
  Serial.begin(115200);
  delay(1000);
  Serial.println(F("\n[BOTTOM] Zigbee Coordinator Honeypot Starting"));
  
  // Setup UART to TOP board
  Serial1.begin(ZB_UART_BAUD, SERIAL_8N1, ZB_UART_RX, ZB_UART_TX);
  
  delay(5000);  // Give TOP board time to boot
  
  // Send boot message to TOP board
  StaticJsonDocument<128> boot;
  boot["evt"]  = "boot";
  boot["role"] = "zigbee_coordinator";
  boot["msg"] = "Honeypot ready";
  sendJson(boot);
  
  // Send hello message
  sendEvt("hello", [](JsonDocument& j){
    j["msg"]  = "hello world";
    j["from"] = "bottom";
  });
  
  // Start Zigbee task in separate thread
  xTaskCreate(esp_zb_task, "Zigbee_main", 4096, NULL, 5, NULL);
}

void loop(){
  // Check for commands from TOP board
  pumpSerial1();

  // Blink LED every 500ms to show we're running
  static uint32_t lastBlink = 0;
  static bool ledState = false;
  if (millis() - lastBlink > 500) {
    lastBlink = millis();
    ledState = !ledState;
    digitalWrite(LED_BUILTIN, ledState);
  }
  
  delay(20);  // Small delay to prevent busy-wait
}