/*
  ESP32-C6 (Seeed XIAO, TOP) — HTTP + MQTT + BLE + Zigbee UART Logger
  - Keeps your existing HTTP/MQTT/BLE honeypot logger.
  - NEW: listens to BOTTOM Zigbee router over Serial1 (D6/D7) as JSON-per-line.
  - NEW: /zigbee.json endpoint and a "Zigbee events" table on the dashboard.
  - NEW: control endpoints to send Zigbee commands to the BOTTOM board.
  Author: Erik Wright
*/

#include <WiFi.h>
#include <WebServer.h>
#include <PubSubClient.h>
#include <NimBLEDevice.h>
#include <ctype.h>
#include <time.h>
#include <ArduinoJson.h>

// ===== Zigbee UART (to bottom board) =====
#define ZB_UART_RX   D7    // XIAO D7 (RX)
#define ZB_UART_TX   D6    // XIAO D6 (TX)
#define ZB_UART_BAUD 115200

// ===== Wi-Fi =====
const char* SSID = "WifiSSIDHere";
const char* PASS = "WifiPassWordHere";

// ===== MQTT (non-TLS) =====
const char* MQTT_HOST = "BrokerIPAddrHere";
const uint16_t MQTT_PORT = 1883;
const char* MQTT_USER = "";
const char* MQTT_PASS = "";

// Topics (built from MAC)
String baseTopic, subTopic, echoTopic;

WiFiClient wifiClient;
PubSubClient mqtt(wifiClient);
WebServer server(80);

// ===== Time / Timestamps =====
static bool g_timeReady = false;

static void setupTimeOnce() {
  setenv("TZ", "CST6CDT,M3.2.0/2,M11.1.0/2", 1);
  tzset();
  configTime(0, 0, "pool.ntp.org", "time.nist.gov", "time.google.com");
  for (int i = 0; i < 20; ++i) {
    time_t now; time(&now);
    if (now > 1700000000) { g_timeReady = true; break; }
    delay(100);
  }
}
static void ensureTime() {
  if (g_timeReady) return;
  time_t now; time(&now);
  if (now > 1700000000) g_timeReady = true;
}
static String currentTimestamp() {
  time_t now; time(&now);
  if (now <= 1700000000) return String("unsynced");
  struct tm localt; localtime_r(&now, &localt);
  char buf[40];
  strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S %Z", &localt);
  return String(buf);
}

// ===== Tiny ring buffer =====
template <typename T, int N>
class Ring {
public:
  void push(const T& v){ buf[i]=v; i=(i+1)%N; if(n<N) n++; }
  bool getNewest(int k, T& out) const { if(k>=n) return false; int j=(i-1-k+N)%N; out=buf[j]; return true; }
  int size() const { return n; }
private: T buf[N]; int i=0, n=0;
};

// ===== HTTP log =====
struct Hit { uint32_t t_ms; String ts; String method, path, ip; };
Ring<Hit, 128> httpLog;
void logHttp(const String& m, const String& p, const String& ip) {
  ensureTime();
  httpLog.push(Hit{ millis(), currentTimestamp(), m, p, ip });
}

// ===== MQTT log =====
struct MQEvt { uint32_t t_ms; String ts; String type; String topic; String payload; };
Ring<MQEvt, 128> mqttLog;
void logMqtt(const String& type, const String& topic, const String& payload="") {
  ensureTime();
  mqttLog.push(MQEvt{ millis(), currentTimestamp(), type, topic, payload });
}

// ===== BLE log =====
struct BLEEvt {
  uint32_t t_ms;
  String ts;
  String type;
  String peer;
  String detail;
  String extra;
  int    rssi;
};
Ring<BLEEvt, 256> bleLog;

static void logBle(const String& type, const String& peer, const String& detail, const String& extra="", int rssi=0) {
  ensureTime();
  bleLog.push(BLEEvt{ millis(), currentTimestamp(), type, peer, detail, extra, rssi });
}

// ===== Helpers =====
static String toHex(const std::string& v, size_t maxBytes=32) {
  const char* hex = "0123456789ABCDEF";
  String s; size_t n = v.size(); if (n > maxBytes) n = maxBytes;
  for (size_t i=0; i<n; ++i) {
    uint8_t b = (uint8_t)v[i];
    s += hex[b>>4]; s += hex[b&0x0F];
    if (i+1<n) s += ":";
  }
  if (v.size() > maxBytes) s += " …";
  return s;
}
static String toAsciiPreview(const std::string& v, size_t maxChars=48) {
  String s; size_t n = v.size(); if (n > maxChars) n = maxChars;
  for (size_t i=0;i<n;++i) {
    char c = (char)v[i];
    if (c >= 32 && c <= 126) s += c; else s += '.';
  }
  if (v.size() > maxChars) s += " …";
  return s;
}

// ===== Globals for display =====
String g_wifiMac, g_clientId;
bool   g_mqttConnected = false;

// ===== BLE globals =====
NimBLEServer*      g_bleServer = nullptr;
NimBLEAdvertising* g_bleAdv    = nullptr;
static NimBLECharacteristic* g_stateChr = nullptr;
static String g_tail;

// ===== Zigbee log (from bottom board) =====
struct ZBEvt {
  uint32_t t_ms;
  String ts;
  String evt;
  String json;
};
Ring<ZBEvt, 256> zbLog;

static String g_zbIn;

static void logZb(const String& evt, const String& json) {
  ensureTime();
  zbLog.push(ZBEvt{ millis(), currentTimestamp(), evt, json });
}

// Serialize and send a single-line JSON to the bottom board
static void zbSend(const JsonDocument& d){
  serializeJson(d, Serial1);
  Serial1.print('\n');
}

// Read lines from Serial1, parse JSON, push into zbLog
static void pumpZigbeeSerial(){
  while (Serial1.available()){
    char c = (char)Serial1.read();
    
    if (c == '\n'){
      StaticJsonDocument<512> doc;
      DeserializationError err = deserializeJson(doc, g_zbIn);
      if (err) {
        logZb("error", String("{\"msg\":\"bad json\",\"detail\":\"")+err.c_str()+"\"}");
      } else {
        String evt = doc["evt"] | String("unknown");
        String compact; serializeJson(doc, compact);
        logZb(evt, compact);
      }
      g_zbIn = "";
    } else if (c != '\r') {
      if (g_zbIn.length() < 700) g_zbIn += c;
    }
  }
}

// ===== Wi-Fi =====
void connectWiFi() {
  WiFi.mode(WIFI_STA);
  WiFi.setHostname("xiao-c6-http-mqtt-ble-zb");
  WiFi.begin(SSID, PASS);
  Serial.print(F("Connecting to ")); Serial.print(SSID);
  unsigned long t0 = millis();
  while (WiFi.status() != WL_CONNECTED && millis() - t0 < 20000) { delay(250); Serial.print("."); }
  Serial.println();
  if (WiFi.status() == WL_CONNECTED) {
    Serial.println(F("WiFi connected"));
    Serial.print(F("IP: ")); Serial.println(WiFi.localIP());
  } else {
    Serial.println(F("WiFi timeout (will retry in loop)."));
  }
  g_wifiMac = WiFi.macAddress();
  Serial.print(F("WiFi MAC: ")); Serial.println(g_wifiMac);

  String mac = g_wifiMac; mac.replace(":", "");
  g_tail = mac.substring(mac.length()-6);
  baseTopic = "esp32c6/" + g_tail;
  subTopic  = baseTopic + "/in/#";
  echoTopic = baseTopic + "/out/echo";
  g_clientId = "xiao-c6-" + g_tail;
}

// ===== MQTT callback/connect =====
void onMqttMessage(char* topic, byte* payload, unsigned int length) {
  String msg; for (unsigned int i=0;i<length && i<160;++i) msg += (char)payload[i];
  if (length>160) msg += " …";
  logMqtt("rx", String(topic), msg);
  if (mqtt.connected()) { mqtt.publish(echoTopic.c_str(), msg.c_str(), false); logMqtt("tx", echoTopic, msg); }
}
bool mqttConnect() {
  mqtt.setServer(MQTT_HOST, MQTT_PORT);
  mqtt.setCallback(onMqttMessage);
  String willTopic = baseTopic + "/status";
  const char* willMsg = "offline";
  bool ok = (MQTT_USER && *MQTT_USER)
              ? mqtt.connect(g_clientId.c_str(), MQTT_USER, MQTT_PASS, willTopic.c_str(), 0, true, willMsg)
              : mqtt.connect(g_clientId.c_str(), willTopic.c_str(), 0, true, willMsg);
  if (ok) {
    g_mqttConnected = true;
    logMqtt("connect", MQTT_HOST);
    mqtt.publish(willTopic.c_str(), "online", true);
    if (mqtt.subscribe(subTopic.c_str())) logMqtt("subscribe", subTopic);
    mqtt.publish(echoTopic.c_str(), "hello from ESP32-C6", false);
    logMqtt("tx", echoTopic, "hello from ESP32-C6");
  } else {
    g_mqttConnected = false;
    logMqtt("error", "connect_failed", String(mqtt.state()));
  }
  return ok;
}

// ===== BLE callbacks =====
class MyServerCallbacks : public NimBLEServerCallbacks {
public:
  void onConnect(NimBLEServer* pServer) { logBle("connect", "-", "conn"); }
  void onDisconnect(NimBLEServer* pServer) { logBle("disconnect", "-", "reason=unknown"); NimBLEDevice::startAdvertising(); }
  void onConnect(NimBLEServer* pServer, ble_gap_conn_desc* desc) {
    String who = desc ? String(NimBLEAddress(desc->peer_ota_addr).toString().c_str()) : String("-");
    logBle("connect", who, "conn+desc");
  }
  void onDisconnect(NimBLEServer* pServer, ble_gap_conn_desc* desc) {
    String who = desc ? String(NimBLEAddress(desc->peer_ota_addr).toString().c_str()) : String("-");
    logBle("disconnect", who, "disc+desc");
    NimBLEDevice::startAdvertising();
  }
};
class LogCharCallbacks : public NimBLECharacteristicCallbacks {
public:
  void onRead(NimBLECharacteristic* pChr) {
    logBle("read", "-", String(pChr->getUUID().toString().c_str()));
  }
  void onWrite(NimBLECharacteristic* pChr) {
    std::string v = pChr->getValue();
    String hex = toHex(v);
    String txt = toAsciiPreview(v);
    logBle("write", "-", String(pChr->getUUID().toString().c_str()), "hex="+hex+"  txt=\""+txt+"\"");
    if (g_stateChr && pChr == g_stateChr) {
      g_stateChr->notify();
      logBle("notify", "-", String(pChr->getUUID().toString().c_str()), "hex="+hex+"  txt=\""+txt+"\"");
    }
  }
  void onRead(NimBLECharacteristic* pChr, NimBLEConnInfo& connInfo) {
    String who = String(connInfo.getAddress().toString().c_str());
    logBle("read", who, String(pChr->getUUID().toString().c_str()));
  }
  void onWrite(NimBLECharacteristic* pChr, NimBLEConnInfo& connInfo) {
    std::string v = pChr->getValue();
    String who = String(connInfo.getAddress().toString().c_str());
    String hex = toHex(v);
    String txt = toAsciiPreview(v);
    logBle("write", who, String(pChr->getUUID().toString().c_str()), "hex="+hex+"  txt=\""+txt+"\"");
    if (g_stateChr && pChr == g_stateChr) {
      g_stateChr->notify();
      logBle("notify", who, String(pChr->getUUID().toString().c_str()), "hex="+hex+"  txt=\""+txt+"\"");
    }
  }
};

// ===== BLE setup =====
void setupBLE() {
  NimBLEDevice::init("Honey-LightBLE");
  NimBLEDevice::setPower(ESP_PWR_LVL_P3);

  g_bleServer = NimBLEDevice::createServer();
  g_bleServer->setCallbacks(new MyServerCallbacks());

  NimBLEService* svcInfo = g_bleServer->createService("180A");
  NimBLECharacteristic* chModel = svcInfo->createCharacteristic("2A24", NIMBLE_PROPERTY::READ);
  NimBLECharacteristic* chSerial= svcInfo->createCharacteristic("2A25", NIMBLE_PROPERTY::READ);
  chModel->setValue("Honey-Light v2");
  chSerial->setValue(("HL-" + g_tail).c_str());
  chModel->setCallbacks(new LogCharCallbacks());
  chSerial->setCallbacks(new LogCharCallbacks());
  svcInfo->start();

  NimBLEService* svcBatt = g_bleServer->createService("180F");
  NimBLECharacteristic* chBatt = svcBatt->createCharacteristic("2A19", NIMBLE_PROPERTY::READ);
  uint8_t batt = 47; chBatt->setValue(&batt, 1);
  chBatt->setCallbacks(new LogCharCallbacks());
  svcBatt->start();

  NimBLEService* svcCfg = g_bleServer->createService("6e400001-b5a3-f393-e0a9-e50e24dcca9e");
  NimBLECharacteristic* chCfg = svcCfg->createCharacteristic(
      "6e400002-b5a3-f393-e0a9-e50e24dcca9e",
      NIMBLE_PROPERTY::READ | NIMBLE_PROPERTY::WRITE | NIMBLE_PROPERTY::WRITE_NR
  );
  chCfg->setValue("cfg=mode:auto;key=unset");
  chCfg->setCallbacks(new LogCharCallbacks());

  g_stateChr = svcCfg->createCharacteristic(
      "6e400003-b5a3-f393-e0a9-e50e24dcca9e",
      NIMBLE_PROPERTY::READ | NIMBLE_PROPERTY::WRITE | NIMBLE_PROPERTY::WRITE_NR | NIMBLE_PROPERTY::NOTIFY
  );
  g_stateChr->setValue("ready");
  g_stateChr->setCallbacks(new LogCharCallbacks());

  svcCfg->start();

  NimBLEAdvertisementData advData;
  advData.setName("Honey-LightBLE");
  advData.addServiceUUID("6e400001-b5a3-f393-e0a9-e50e24dcca9e");
  advData.addServiceUUID("180F");

  NimBLEAdvertisementData scanData;
  std::string mfg("\x4C\x00", 2);
  mfg += "HL-"; mfg += g_tail.c_str();
  scanData.setManufacturerData(mfg);

  g_bleAdv = NimBLEDevice::getAdvertising();
  g_bleAdv->setAdvertisementData(advData);
  g_bleAdv->setScanResponseData(scanData);

  NimBLEDevice::startAdvertising();
  logBle("adv", "-", "Honey-LightBLE started");
}

// ===== Web UI =====
String html() {
  String s =
    F("<!doctype html><html><head><meta charset='utf-8'>"
      "<meta name='viewport' content='width=device-width,initial-scale=1'>"
      "<meta http-equiv='refresh' content='2'>"
      "<title>ESP32-C6 Logger (HTTP/MQTT/BLE/ZB)</title>"
      "<style>body{font-family:system-ui;margin:20px}"
      "table{border-collapse:collapse;width:100%}"
      "th,td{border:1px solid #ccc;padding:6px;text-align:left;vertical-align:top}"
      "th{background:#f4f4f4}</style></head><body>"
      "<h2>ESP32-C6 HTTP + MQTT + BLE + Zigbee Logger</h2>");
  s += "<p><b>IP:</b> " + WiFi.localIP().toString()
     + " &middot; <b>AP RSSI:</b> " + String(WiFi.RSSI()) + "</p>";
  s += "<p><b>Wi-Fi MAC:</b> " + g_wifiMac
     + " &middot; <b>MQTT:</b> " + (g_mqttConnected ? String("connected") : String("disconnected"))
     + " &middot; <b>ClientID:</b> " + g_clientId + "</p>";
  s += "<p><b>Base topic:</b> " + baseTopic + " &nbsp; "
       "<a href='/http.json'>http.json</a> &middot; "
       "<a href='/mqtt.json'>mqtt.json</a> &middot; "
       "<a href='/ble.json'>ble.json</a> &middot; "
       "<a href='/zigbee.json'>zigbee.json</a> &middot; "
       "<a href='/ping'>/ping</a></p>";

  s += "<p><b>Time sync:</b> " + String(g_timeReady ? "OK" : "unsynced (showing 'unsynced' until NTP)") + "</p>";

  s += "<h3>Zigbee controls</h3>"
       "<p><a href='/zb/permit?sec=60'>Open Join (60s)</a> &middot; "
       "<a href='/zb/identify?ieee=00124B0012345678&sec=3'>Identify sample</a> &middot; "
       "<a href='/zb/on?ieee=00124B0012345678&ep=1'>On</a> &middot; "
       "<a href='/zb/off?ieee=00124B0012345678&ep=1'>Off</a></p>";

  s += "<h3>Zigbee events (newest first)</h3><table><tr><th>#</th><th>t(ms)</th><th>time</th><th>evt</th><th>payload</th></tr>";
  ZBEvt ze; int nz=0;
  for (; zbLog.getNewest(nz, ze); ++nz) {
    s += "<tr><td>"+String(nz+1)+"</td><td>"+String(ze.t_ms)+"</td><td>"+ze.ts+
         "</td><td>"+ze.evt+"</td><td><code>"+ze.json+"</code></td></tr>";
    if (nz >= 99) break;
  }
  if (nz == 0) s += F("<tr><td colspan='5'>No Zigbee events yet. Try opening join or triggering a device.</td></tr>");
  s += "</table>";

  s += "<h3>BLE events (newest first)</h3><table><tr><th>#</th><th>t(ms)</th><th>time</th><th>type</th><th>peer</th><th>detail</th><th>data</th></tr>";
  BLEEvt be; int nb=0;
  for (; bleLog.getNewest(nb, be); ++nb) {
    s += "<tr><td>"+String(nb+1)+"</td><td>"+String(be.t_ms)+"</td><td>"+be.ts+
         "</td><td>"+be.type+"</td><td>"+be.peer+"</td><td>"+be.detail+"</td><td>"+be.extra+"</td></tr>";
    if (nb >= 99) break;
  }
  if (nb == 0) s += F("<tr><td colspan='7'>No BLE events yet.</td></tr>");
  s += "</table>";

  s += "<h3>MQTT events (newest first)</h3><table><tr><th>#</th><th>t(ms)</th><th>time</th><th>type</th><th>topic</th><th>payload</th></tr>";
  MQEvt me; int nm=0;
  for (; mqttLog.getNewest(nm, me); ++nm) {
    s += "<tr><td>"+String(nm+1)+"</td><td>"+String(me.t_ms)+"</td><td>"+me.ts+
         "</td><td>"+me.type+"</td><td>"+me.topic+"</td><td>"+me.payload+"</td></tr>";
    if (nm >= 49) break;
  }
  if (nm == 0) s += "<tr><td colspan='6'>No MQTT events yet. Publish to <code>"+baseTopic+"/in/test</code>.</td></tr>";
  s += "</table>";

  s += "<h3>HTTP hits (newest first)</h3><table><tr><th>#</th><th>t(ms)</th><th>time</th><th>method</th><th>path</th><th>ip</th></tr>";
  Hit h; int nh=0;
  for (; httpLog.getNewest(nh, h); ++nh) {
    s += "<tr><td>"+String(nh+1)+"</td><td>"+String(h.t_ms)+"</td><td>"+h.ts+
         "</td><td>"+h.method+"</td><td>"+h.path+"</td><td>"+h.ip+"</td></tr>";
    if (nh >= 49) break;
  }
  if (nh == 0) s += F("<tr><td colspan='6'>No HTTP hits yet. Try <a href='/ping'>/ping</a>.</td></tr>");
  s += "</table></body></html>";
  return s;
}

// ===== HTTP handlers =====
void handleHome()     { server.send(200, "text/html", html()); }
void handleHttpJson() {
  String s="["; bool first=true; Hit h;
  for (int i=0; httpLog.getNewest(i,h) && i<127; ++i) {
    if(!first) s+=","; first=false;
    s += "{\"t\":"+String(h.t_ms)+",\"time\":\""+h.ts+"\",\"m\":\""+h.method+"\",\"p\":\""+h.path+"\",\"ip\":\""+h.ip+"\"}";
  }
  s += "]"; server.send(200, "application/json", s);
}
void handleMqttJson() {
  String s="["; bool first=true; MQEvt me;
  for (int i=0; mqttLog.getNewest(i,me) && i<127; ++i) {
    if(!first) s+=","; first=false;
    s += "{\"t\":"+String(me.t_ms)+",\"time\":\""+me.ts+"\",\"type\":\""+me.type+"\",\"topic\":\""+me.topic+"\",\"payload\":\""+me.payload+"\"}";
  }
  s += "]"; server.send(200, "application/json", s);
}
void handleBleJson() {
  String s="["; bool first=true; BLEEvt be;
  for (int i=0; bleLog.getNewest(i,be) && i<255; ++i) {
    if(!first) s+=","; first=false;
    s += "{\"t\":"+String(be.t_ms)+",\"time\":\""+be.ts+"\",\"type\":\""+be.type+"\",\"peer\":\""+be.peer+
         "\",\"detail\":\""+be.detail+"\",\"extra\":\""+be.extra+"\"}";
  }
  s += "]"; server.send(200, "application/json", s);
}
void handleZigbeeJson() {
  String s="["; bool first=true; ZBEvt ze;
  for (int i=0; zbLog.getNewest(i,ze) && i<255; ++i) {
    if(!first) s+=","; first=false;
    s += "{\"t\":"+String(ze.t_ms)+",\"time\":\""+ze.ts+"\",\"evt\":\""+ze.evt+"\",\"payload\":"+ze.json+"}";
  }
  s += "]"; server.send(200, "application/json", s);
}
void handlePing() {
  logHttp("GET","/ping",server.client().remoteIP().toString());
  server.send(200,"text/plain","pong");
}

void handleNotFound() {
  String m=(server.method()==HTTP_GET)?"GET":(server.method()==HTTP_POST)?"POST":(server.method()==HTTP_PUT)?"PUT":(server.method()==HTTP_DELETE)?"DELETE":"OTHER";
  logHttp(m,server.uri(),server.client().remoteIP().toString());
  server.send(404,"text/plain","Not found");
}

// ===== Zigbee control endpoints =====
void handleZbPermit() {
  int sec = server.hasArg("sec") ? server.arg("sec").toInt() : 60;
  StaticJsonDocument<64> d; d["cmd"]="zb_permit_join"; d["seconds"]=sec;
  zbSend(d);
  server.send(200,"application/json", String("{\"ok\":true,\"seconds\":")+sec+"}");
}
void handleZbIdentify() {
  String ieee = server.arg("ieee");
  int sec = server.hasArg("sec") ? server.arg("sec").toInt() : 3;
  if (ieee.length()==0){ server.send(400,"application/json","{\"ok\":false,\"err\":\"missing ieee\"}"); return; }
  StaticJsonDocument<96> d; d["cmd"]="zb_identify"; d["ieee"]=ieee; d["seconds"]=sec;
  zbSend(d);
  server.send(200,"application/json","{\"ok\":true}");
}
void handleZbOnOff(bool on) {
  String ieee = server.arg("ieee");
  int ep = server.hasArg("ep") ? server.arg("ep").toInt() : 1;
  if (ieee.length()==0){ server.send(400,"application/json","{\"ok\":false,\"err\":\"missing ieee\"}"); return; }
  StaticJsonDocument<128> d; d["cmd"]="zb_onoff"; d["ieee"]=ieee; d["ep"]=ep; d["on"]= on?1:0;
  zbSend(d);
  server.send(200,"application/json","{\"ok\":true}");
}
void handleZbOn(){ handleZbOnOff(true); }
void handleZbOff(){ handleZbOnOff(false); }

// ===== Setup / Loop =====
void setup() {
  pinMode(LED_BUILTIN, OUTPUT);
  Serial.begin(115200);
  delay(200);
  
  Serial.println(F("\n[TOP] === Starting ESP32-C6 Honeypot ==="));
  Serial.printf("Free heap: %d bytes\n", ESP.getFreeHeap());

  Serial1.begin(ZB_UART_BAUD, SERIAL_8N1, ZB_UART_RX, ZB_UART_TX);

  connectWiFi();
  Serial.printf("After WiFi - Free heap: %d bytes\n", ESP.getFreeHeap());
  
  setupTimeOnce();
  
  setupBLE();
  Serial.printf("After BLE - Free heap: %d bytes\n", ESP.getFreeHeap());

  server.on("/", handleHome);
  server.on("/http.json", handleHttpJson);
  server.on("/mqtt.json", handleMqttJson);
  server.on("/ble.json", handleBleJson);
  server.on("/zigbee.json", handleZigbeeJson);
  server.on("/ping", HTTP_GET, handlePing);

  server.on("/zb/permit", HTTP_GET, handleZbPermit);
  server.on("/zb/identify", HTTP_GET, handleZbIdentify);
  server.on("/zb/on", HTTP_GET, handleZbOn);
  server.on("/zb/off", HTTP_GET, handleZbOff);

  server.onNotFound(handleNotFound);
  server.begin();
}

void loop() {
  server.handleClient();
  pumpZigbeeSerial();

  // Check if BLE is still running
  static uint32_t lastBleCheck = 0;
  if (millis() - lastBleCheck > 30000) { // Every 30 seconds
    lastBleCheck = millis();
    if (g_bleAdv && !g_bleAdv->isAdvertising()) {
      Serial.println(F("[BLE] WARNING: Advertising stopped! Restarting..."));
      NimBLEDevice::startAdvertising();
    }
  }

  static uint32_t lastConnTry = 0;
  if (WiFi.status() == WL_CONNECTED) {
    if (!mqtt.connected()) {
      g_mqttConnected = false;
      if (millis() - lastConnTry > 5000) {
        lastConnTry = millis();
        mqttConnect();
      }
    } else {
      mqtt.loop();
    }
  }

  static uint32_t lastBlink = 0;
  static bool ledState = false;
  if (millis() - lastBlink > 500) {
    lastBlink = millis();
    ledState = !ledState;
    digitalWrite(LED_BUILTIN, ledState ? HIGH : LOW);
  }

  static uint32_t retry=0;
  if (WiFi.status() != WL_CONNECTED && millis() - retry > 5000) {
    retry = millis();
    WiFi.disconnect(true);
    WiFi.begin(SSID, PASS);
  }
}